
# Insert Custom Column
# Copyright 2006 by Alexander V. Christensen

"""
Creates a user-defined column type and inserts it in the current report.
"""

#    This file is part of GanttPV.
#
#    GanttPV is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    GanttPV is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with GanttPV; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# 081122 - fixed the data type for text columns

import re

def CustomColumn(rtname, label, extras={}):
    name = 'xx' + re.sub('[^a-zA-Z0-9]', '', label) # ignore non-alphanumberics
    if debug: print "custom column name:", name
    column = {'Name': name, 'Label': label, 'DataType': 't', 'AccessType': 'd', 'T': 'A', 'Edit': True, 'Width': 120}
    column.update(extras)
    rt = {'Name': rtname} 
    ct = [column]
    Data.AddReportType(rt, ct)
    return name

def ReinsertColumn(colid, insertAfter=False):
    clist = Data.GetColumnList(self.ReportID)
    sel = self.Report.GetSelectedCols()
    # if debug: print 'selection', sel
    if sel:
        if insertAfter:
            pos = max(sel)
        else:
            pos = min(sel)
    else:
        # pos = self.Report.GetGridCursorCol()
        pos = len(clist)
    if insertAfter: pos += 1  # insert after
    clist.insert(pos, colid)
    Data.ReorderReportColumns(self.ReportID, clist)

defDataTypes = [('Text', 't'), ('Date', 'd'), ('Number', 'f'), ('Time Units', 'u')]

class CustomColumnDialog(wx.Dialog):
    def __init__(self, parent, tables, dataTypes=defDataTypes):
        wx.Dialog.__init__(self, parent, -1, 'Custom Column')
        dataChoices = []
        self.dataTypes = {}
        for choice, dt in dataTypes:
            dataChoices.append(choice)
            self.dataTypes[choice] = dt

        border_sizer = wx.BoxSizer(wx.VERTICAL)
        sizer = wx.BoxSizer(wx.VERTICAL)

        self.labelLabel = wx.StaticText(self, -1, 'Name your new column:')
        sizer.Add(self.labelLabel, 0, wx.ALL | wx.ALIGN_CENTRE, 5)
        # self.labelCtrl = wx.TextCtrl(self, -1, size=(200, 40), style=wx.TE_MULTILINE)
        self.labelCtrl = wx.TextCtrl(self, -1, size=(200, 25))
        self.labelCtrl.Bind(wx.EVT_SET_FOCUS, self.OnFocus)
        label = Data.Option.get('CustomColumnLabel')
        if label:
            self.labelCtrl.SetValue(label)
        sizer.Add(self.labelCtrl, 0, wx.ALL | wx.ALIGN_CENTRE, 5)

        midSizer = wx.BoxSizer(wx.HORIZONTAL)
        labelSizer = wx.BoxSizer(wx.VERTICAL)  # left side
        ctrlSizer = wx.BoxSizer(wx.VERTICAL)  # right side

        self.tableChoice = (len(tables) > 1)
        tableCtrlLabel = wx.StaticText(self, -1, "Table:", style=wx.ALIGN_CENTRE)
        labelSizer.Add(tableCtrlLabel, 0, wx.ALL | wx.ALIGN_RIGHT, 5)
        if self.tableChoice:
            self.tableCtrl = wx.Choice(self, -1, choices=tables)
            table = Data.Option.get('CustomColumnTable')
            if not table or not self.tableCtrl.SetStringSelection(table):
                self.tableCtrl.SetSelection(0)
            ctrlSizer.Add(self.tableCtrl, 0, wx.ALL | wx.ALIGN_LEFT, 5)
        else:
            self.tableCtrl = wx.StaticText(self, -1, tables[0])
            ctrlSizer.Add(self.tableCtrl, 0, wx.ALL | wx.ALIGN_LEFT, 5)

        typeCtrlLabel = wx.StaticText(self, -1, "Data type:", style=wx.ALIGN_CENTRE)
        labelSizer.Add(typeCtrlLabel, 0, wx.ALL | wx.ALIGN_RIGHT, 5)
        self.typeCtrl = wx.Choice(self, -1, choices=dataChoices)
        type = Data.Option.get('CustomColumnDataType')
        if not type or not self.typeCtrl.SetStringSelection(type):
            self.typeCtrl.SetSelection(0)
        ctrlSizer.Add(self.typeCtrl, 0, wx.ALL | wx.ALIGN_LEFT, 5)

        midSizer.Add(labelSizer, 0, wx.ALL | wx.ALIGN_CENTRE, 5) 
        midSizer.Add(ctrlSizer, 0, wx.ALL | wx.ALIGN_CENTRE, 5) 
        sizer.Add(midSizer, 0, wx.ALL | wx.ALIGN_CENTRE, 5)
        sizer.Add((25, 25))
        buttons = wx.BoxSizer(wx.HORIZONTAL)

        cancelbutton = wx.Button(self, wx.ID_CANCEL, "Cancel", size=(70, 20))
        buttons.Add(cancelbutton, 0, wx.RIGHT, 5)

        okbutton = wx.Button(self, wx.ID_OK, "OK", size=(70, 20))
        okbutton.SetDefault()
        okbutton.Bind(wx.EVT_BUTTON, self.OnOkay)
        buttons.Add(okbutton, 0, wx.LEFT, 5)

        sizer.Add(buttons, 0, wx.ALIGN_RIGHT)

        border_sizer.Add(sizer, 0, wx.ALL | wx.ALIGN_CENTRE, 10)
        self.SetSizer(border_sizer)
        self.Fit()
        self.CentreOnScreen()

    def OnFocus(self, evt):
        source = evt.GetEventObject()
        if isinstance(source, wx.TextCtrl):
             source.SetSelection(-1, -1)
        evt.Skip()
        self.Refresh()

    def GetLabel(self):
        return self.labelCtrl.GetValue()

    def GetTable(self):
        if self.tableChoice:
            return self.tableCtrl.GetStringSelection()
        return self.tableCtrl.GetLabel()

    def GetTableNumber(self):
        if self.tableChoice:
            return self.tableCtrl.GetSelection()
        return 0

    def GetDataType(self):
        typeString = self.typeCtrl.GetStringSelection()
        return self.dataTypes[typeString]

    def OnOkay(self, evt):
        Data.Option['CustomColumnLabel'] = self.GetLabel()
        if self.tableChoice:
            Data.Option['CustomColumnTable'] = self.GetTable()
        Data.Option['CustomColumnDataType'] = self.typeCtrl.GetStringSelection()
        Data.SaveOption()
        evt.Skip()

def Do():
    rt = self.Report.table.reporttype
    ta = rt.get('TableA')
    tb = rt.get('TableB')
    if tb:
        tables = [ta, tb]
    else:
        tables = [ta]
    dlg = CustomColumnDialog(None, tables)
    if dlg.ShowModal() != wx.ID_OK:
        return

    label = dlg.GetLabel()
    if not label:
        return
    dt = dlg.GetDataType()
    width = {'t': 120, 'd': 80, 'f': 60, 'u': 60}.get(dt, 120)
    T = 'AB'[dlg.GetTableNumber()]
    also = rt.get('Also')
    if T == 'A' and also:
        rtname = Data.ReportType.get(also, {}).get('Name')
    else:
        rtname = rt.get('Name')
    extras = {'DataType': dt, 'T': T, 'Width': width}
    ctname = CustomColumn(rtname, label, extras=extras)
    colid = Data.AddColumn(self.ReportID, {'ColumnTypeID': (rtname, ctname)})
    ReinsertColumn(colid)
    Data.SetUndo("Insert New Column")

Do()
